// Основные импорты Flutter и нужных пакетов
import 'package:flutter/material.dart';
import 'package:flutter/services.dart';
import 'package:flutter_inappwebview/flutter_inappwebview.dart'; // Для отображения сайта внутри приложения
import 'package:connectivity_plus/connectivity_plus.dart'; // Для проверки наличия интернета
import 'package:flutter/cupertino.dart'; // Для iOS-стиль кнопки
import 'package:flutter_localizations/flutter_localizations.dart'; // Для поддержки локализации
import 'package:firebase_core/firebase_core.dart'; // Для инициализации Firebase
import 'firebase_options.dart'; // Для передачи параметров инициализации из сгенерированного файла
import 'package:firebase_messaging/firebase_messaging.dart'; // Для работы с push-уведомлениями
import 'dart:convert'; // Для безопасной передачи токена в JS
import 'package:geolocator/geolocator.dart'; // Для получения GPS-координат

// Точка входа в приложение
// Функция для обработки push-уведомлений в фоновом режиме
Future<void> _firebaseMessagingBackgroundHandler(RemoteMessage message) async {
  // Обязательно инициализируем Firebase в фоне
  await Firebase.initializeApp();
  // Здесь можно обработать данные уведомления
  // print('Получено фоновое push-уведомление: ${message.messageId}');
}

/// Точка входа в приложение. Здесь происходит вся инициализация Flutter и Firebase.
/// Важно: обязательно вызываем WidgetsFlutterBinding.ensureInitialized() до любых async-операций.
Future<void> main() async {
  WidgetsFlutterBinding.ensureInitialized();
  // Инициализация Firebase с параметрами из firebase_options.dart (обязательно для новых версий!)
  await Firebase.initializeApp(
    options: DefaultFirebaseOptions.currentPlatform,
  );
  // Устанавливаем цвет статусбара (#67A900) и белые иконки
  SystemChrome.setSystemUIOverlayStyle(const SystemUiOverlayStyle(
    statusBarColor: Color(0xFF67A900), // Цвет статусбара
    statusBarIconBrightness: Brightness.light, // Белые иконки для Android
    statusBarBrightness: Brightness.light, // Белые иконки для iOS
  ));
  // Регистрируем обработчик фоновых push-уведомлений (важно вызывать после инициализации Firebase)
  FirebaseMessaging.onBackgroundMessage(_firebaseMessagingBackgroundHandler);
  // Запускаем основное приложение
  runApp(const MyApp());
}

// Главный виджет приложения
class MyApp extends StatelessWidget {
  const MyApp({super.key});

  @override
  Widget build(BuildContext context) {
    return MaterialApp(
      debugShowCheckedModeBanner: false, // Убираем лейбл DEBUG
      title: 'BXSTORE',
      theme: ThemeData(
        colorScheme: ColorScheme.fromSeed(seedColor: const Color(0xFF67A900)),
        appBarTheme: const AppBarTheme(
          backgroundColor: Color(0xFF67A900),
          foregroundColor: Colors.white,
          iconTheme: IconThemeData(color: Colors.white),
          titleTextStyle: TextStyle(color: Colors.white, fontSize: 20, fontWeight: FontWeight.w600),
        ),
      ),
      localizationsDelegates: const [
        GlobalMaterialLocalizations.delegate,
        GlobalWidgetsLocalizations.delegate,
        GlobalCupertinoLocalizations.delegate,
      ],
      supportedLocales: const [
        Locale('en'),
        Locale('ru'),
      ],
      localeResolutionCallback: (locale, supportedLocales) {
        if (locale == null) return const Locale('en');
        for (var supportedLocale in supportedLocales) {
          if (supportedLocale.languageCode == locale.languageCode) {
            return supportedLocale;
          }
        }
        return const Locale('en');
      },
      home: const MyHomePage(),
    );
  }
}

// Главный экран приложения (StatefulWidget, т.к. нужно отслеживать состояние интернета)
class MyHomePage extends StatefulWidget {
  const MyHomePage({super.key});

  @override
  State<MyHomePage> createState() => _MyHomePageState();
}

class _MyHomePageState extends State<MyHomePage> {
  // Переменные для хранения координат
  double? _latitude;
  double? _longitude;
  // Флаг, что данные уже переданы в JS
  bool _dataSent = false;

  // Получение координат устройства
  Future<void> _getLocation() async {
    bool serviceEnabled;
    LocationPermission permission;

    // Проверяем, включен ли сервис геолокации
    serviceEnabled = await Geolocator.isLocationServiceEnabled();
    if (!serviceEnabled) {
      // Геолокация выключена, можно показать диалог пользователю
      return;
    }

    // Проверяем разрешения
    permission = await Geolocator.checkPermission();
    if (permission == LocationPermission.denied) {
      permission = await Geolocator.requestPermission();
      if (permission == LocationPermission.denied) {
        // Нет разрешения
        return;
      }
    }
    if (permission == LocationPermission.deniedForever) {
      // Разрешение навсегда запрещено
      return;
    }

    // Получаем текущие координаты
    Position position = await Geolocator.getCurrentPosition();
    setState(() {
      _latitude = position.latitude;
      _longitude = position.longitude;
    });

    // Если WebView открыт — передаём координаты в JS
    if (_webViewController != null) {
      _sendLocationToWeb(_latitude!, _longitude!);
    }
  }

  // Метод для однократной передачи FCM токена
  void _sendTokenToWeb(String token) {
    if (!_dataSent) {
      final safeToken = jsonEncode(token);
      _webViewController?.evaluateJavascript(
        source: "window.setFcmToken?.($safeToken)"
      ).catchError((e) {
        debugPrint('Token send error: $e');
      });
    }
  }

  // Метод для однократной передачи координат в WebView
  void _sendLocationToWeb(double lat, double lng) {
    if (!_dataSent) {
      final safeLat = jsonEncode(lat);
      final safeLng = jsonEncode(lng);
      _webViewController?.evaluateJavascript(
        source: "window.setDeviceLocation?.($safeLat, $safeLng)"
      ).catchError((e) {
        debugPrint('Location send error: $e');
      });
    }
  }

  // Переменная для хранения состояния интернет-соединения
  bool? _hasConnection;
  // Токен устройства для push-уведомлений
  String? _fcmToken;
  // Контроллер для управления WebView
  InAppWebViewController? _webViewController;

  @override
  void initState() {
    super.initState();
    // Проверяем интернет при запуске экрана
    _checkConnection();
    // Инициализируем push-уведомления
    _initFirebaseMessaging();
    // Получаем координаты устройства
    _getLocation();
  }

  // Метод для проверки наличия интернет-соединения
  Future<void> _checkConnection() async {
    final connectivityResult = await Connectivity().checkConnectivity();
    setState(() {
      _hasConnection = connectivityResult != ConnectivityResult.none;
    });
  }

  // Метод для инициализации push-уведомлений
  Future<void> _initFirebaseMessaging() async {
    FirebaseMessaging messaging = FirebaseMessaging.instance;

    // Запрашиваем разрешение на уведомления (особенно важно для iOS)
    await messaging.requestPermission();

    // Получаем токен устройства (его можно отправить на сервер для рассылки push)
    String? token = await messaging.getToken();
    setState(() {
      _fcmToken = token;
    });
    // Добавим логирование и popup для токена
    debugPrint('FCM TOKEN: '
        // ignore: unnecessary_string_interpolations
        '${token ?? "(null)"}');
    if (token != null) {
      // Покажем алерт на экране (если токен есть)
      try {
        // ignore: use_build_context_synchronously
        showDialog(
          context: context,
          builder: (ctx) => AlertDialog(
            title: const Text('FCM TOKEN'),
            content: Text(token),
            actions: [
              TextButton(
                onPressed: () => Navigator.of(ctx).pop(),
                child: const Text('OK'),
              ),
            ],
          ),
        );
      } catch (e) {
        // В лог, если context недоступен (например, в фоне): просто игнорируем
        debugPrint('FCM TOKEN ALERT ERROR: $e');
      }
    }

    // Если WebView уже открыт — передаём токен в JS
    if (_webViewController != null && token != null) {
      _sendTokenToWeb(token);
    }

    // Обработка push-уведомлений в foreground
    FirebaseMessaging.onMessage.listen((RemoteMessage message) {
      // Здесь можно показать локальное уведомление или обработать данные
    });

    // Обработка нажатия на уведомление (background/opened)
    FirebaseMessaging.onMessageOpenedApp.listen((RemoteMessage message) {
      // Здесь можно реализовать переход на нужный экран
    });
  }

  // (Передача токена в WebView отключена по требованию)

  @override
  Widget build(BuildContext context) {
    // Если еще не знаем, есть ли интернет — показываем индикатор загрузки
    if (_hasConnection == null) {
      return Scaffold(
        backgroundColor: Color(0xFF67A900),
        body: Center(child: CircularProgressIndicator()),
      );
    }
    // Если интернет есть — показываем сайт внутри приложения через WebView
    if (_hasConnection == true) {
      return Scaffold(
        backgroundColor: Color(0xFF67A900),
        body: SafeArea(
          child: InAppWebView(
            initialUrlRequest: URLRequest(url: WebUri('https://bxstore.ru/mobileapp/')),
            initialSettings: InAppWebViewSettings(
              javaScriptEnabled: true,
              allowsInlineMediaPlayback: true,
              mediaPlaybackRequiresUserGesture: false,
              useOnDownloadStart: true,
              allowsBackForwardNavigationGestures: true,
              userAgent: 'BXSTORE',
            ),
            onWebViewCreated: (controller) {
              _webViewController = controller;
              // Передаём токен и координаты, если они есть (на всякий случай сразу, как только контроллер готов)
              if (_fcmToken != null) _sendTokenToWeb(_fcmToken!);
              if (_latitude != null && _longitude != null) _sendLocationToWeb(_latitude!, _longitude!);
            },
            onLoadStop: (controller, url) {
              if (!_dataSent) {
                // Передаём данные только при первом успешном LoadStop
                if (_fcmToken != null) _sendTokenToWeb(_fcmToken!);
                if (_latitude != null && _longitude != null) _sendLocationToWeb(_latitude!, _longitude!);
                _dataSent = true;
              }
            },
            // Можно добавить обработку ошибок загрузки
            onLoadError: (controller, url, code, message) {
              debugPrint('WebView load error: $code $message');
            },
          ),
        ),
      );
    } else {
      // Если интернета нет — показываем сообщение и кнопку "Попробовать снова"
      return Scaffold(
        backgroundColor: Color(0xFF67A900),
        body: Center(
          child: Column(
            mainAxisAlignment: MainAxisAlignment.center,
            children: [
              Text(
                Localizations.localeOf(context).languageCode == 'ru'
                    ? 'Интернет соединение отсутствует'
                    : 'No internet connection',
                style: const TextStyle(fontSize: 18),
                textAlign: TextAlign.center,
              ),
              const SizedBox(height: 24),
              SizedBox(
                width: 200,
                height: 48,
                child: CupertinoButton(
                  color: Colors.black,
                  borderRadius: BorderRadius.all(Radius.circular(12)),
                  child: Text(
                    Localizations.localeOf(context).languageCode == 'ru'
                        ? 'Попробовать снова'
                        : 'Try again',
                    style: const TextStyle(color: Colors.white, fontSize: 16),
                  ),
                  onPressed: _checkConnection,
                ),
              ),
            ],
          ),
        ),
      );
    }
  }
}